<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Model;

class Karpet extends Model
{
    protected $table = 'karpet';

    protected $fillable = [
        'nama',
        'kategori',
        'ketebalan',
        'stok_per_meter',
        'minimum_stok_per_meter',
        'stok_per_roll',
        'minimum_stok_per_roll',
        'harga_per_meter',
        'foto',
        'deskripsi'
    ];

    protected $casts = [
        'stok_per_meter' => 'integer',
        'minimum_stok_per_meter' => 'integer',
        'stok_per_roll' => 'integer',
        'minimum_stok_per_roll' => 'integer',
        'harga_per_meter' => 'integer',
    ];

    // Accessor untuk format harga
    protected function hargaFormatted(): Attribute
    {
        return Attribute::make(
            get: fn() => 'Rp ' . number_format($this->harga_per_meter, 0, ',', '.')
        );
    }

    // Accessor untuk cek stok per meter menipis
    protected function isStokPerMeterMenipis(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->stok_per_meter <= $this->minimum_stok_per_meter
        );
    }

    // Accessor untuk cek stok per roll menipis
    protected function isStokPerRollMenipis(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->stok_per_roll <= $this->minimum_stok_per_roll
        );
    }

    // Accessor untuk cek apakah ada stok yang menipis (salah satu atau keduanya)
    protected function isStokMenipis(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->is_stok_per_meter_menipis || $this->is_stok_per_roll_menipis
        );
    }

    // Accessor untuk foto URL
    protected function fotoUrl(): Attribute
    {
        return Attribute::make(
            get: fn() => $this->foto ? asset($this->foto) : null
        );
    }

    // Scope untuk filter
    public function scopeFilter($query, $search = null, $kategori = null, $ketebalan = null)
    {
        return $query
            ->when($search, fn($q) => $q->where('nama', 'like', "%{$search}%"))
            ->when($kategori, fn($q) => $q->where('kategori', $kategori))
            ->when($ketebalan, fn($q) => $q->where('ketebalan', $ketebalan));
    }

    // Scope untuk stok menipis
    public function scopeStokMenipis($query)
    {
        return $query->where(function ($q) {
            $q->whereRaw('stok_per_meter <= minimum_stok_per_meter')
                ->orWhereRaw('stok_per_roll <= minimum_stok_per_roll');
        });
    }

    // Static method untuk get unique values
    public static function getKategoris()
    {
        return static::distinct()->pluck('kategori')->filter()->sort()->values();
    }

    public static function getKetebalan()
    {
        return static::distinct()->pluck('ketebalan')->filter()->sort()->values();
    }
}
